//*************************************************************************************************
//
//	Description:
//		msaa.fx - PS3 MSAA shader for 4X & 2X
//
//	<P> Copyright (c) 2007 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tim Mann


#include "stddefs.fxh"

//
// Textures
//

texture Texture4X : TEXTURE < bool appEdit = true; >;
texture Texture2X1 : TEXTURE < bool appEdit = true; >;
texture Texture2X2 : TEXTURE < bool appEdit = true; >;
float4 ClearColour <bool adppEdit=false; >;

//-----------------------------------------------------------------------
//
// Samplers
//


sampler2D Map4X : SAMPLER < bool appEdit = false; string SamplerTexture="Texture4X"; > = sampler_state
{
	Texture = < Texture4X >;
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Linear;
	MagFilter = Linear;
	AddressU = ClampToEdge;
	AddressV = ClampToEdge;
};

sampler2D Map2X1 : SAMPLER < bool appEdit = false; string SamplerTexture="Texture2X1"; > = sampler_state
{
	Texture = < Texture2X1 >;
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = ConvolutionMin;
	MagFilter = ConvolutionMag;
	ConvFilter = Quincunx;
	AddressU = ClampToEdge;
	AddressV = ClampToEdge;
};

sampler2D Map2X2 : SAMPLER < bool appEdit = false; string SamplerTexture="Texture2X2"; > = sampler_state
{
	Texture = < Texture2X2 >;
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = ConvolutionMin;
	MagFilter = ConvolutionMag;
	ConvFilter = QuincunxAlt;
	AddressU = ClampToEdge;
	AddressV = ClampToEdge;
};

// Post processing effects
#include "PostProcessingFilters.fxh"

//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float2 position : POSITION;
};


// Output structure (also input to the fragment shader)
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
};

struct VSOUTPUT2X
{
	float4 position			: POSITION;													// View-coords position
	float4 texCoords	: TEXCOORD0;												// UV coords for texture channel 0 and 1 (quincunx offset)
};

struct VSCLEAROUTPUT
{
	float4 position		: POSITION;													// View-coords position
};

//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT MSAAVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

	_output.texCoord.x = _input.position.x*0.5f+0.5f;
	_output.texCoord.y = _input.position.y*-0.5f+0.5f;
	_output.position = float4(_input.position.xy, 0.0f, 1.0f);

	return _output;
}
			
VSOUTPUT2X MSAAVertexShader2X( VSINPUT _input )
{
	VSOUTPUT2X _output;

	_output.texCoords.x = _input.position.x*0.5f+0.5f;
	_output.texCoords.y = _input.position.y*-0.5f+0.5f;
	_output.texCoords.zw = _output.texCoords.xy;
	_output.texCoords += float4( 0.0001f, 0.0f, -0.0001f, -0.001f );
	_output.position = float4(_input.position.xy, 0.0f, 1.0f);

	return _output;
}

VSCLEAROUTPUT MSAAClearVertexShader( VSINPUT _input )
{
	VSCLEAROUTPUT _output;

	_output.position = float4(_input.position.xy, 0.0f, 1.0f);

	return _output;
}


//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};


//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT FragmentShader4X( VSOUTPUT _input )
{
	PSOUTPUT _output;

	// Read textures
	float2	uv = _input.texCoord;
	half4	Colour;
#ifdef _POST_PARSE_
  	Colour = h4tex2D( Map4X, uv );
#else
  	Colour = tex2D( Map4X, uv );
#endif

#if defined(POST_PROCESS_EFFECT_INDEX)
	_output.Colour.rgb = DoSimplePostProcessingEffect( Colour.rgb, _input.texCoord );
	_output.Colour.a=Colour.a;
#else
	_output.Colour = Colour;
#endif

	return _output;
}

PSOUTPUT FragmentShader2X( VSOUTPUT2X _input )
{
	PSOUTPUT _output;
	half4	Colour;

	float4	uvs = _input.texCoords;

#ifdef _POST_PARSE_
    half4 color0 = h4tex2D(Map2X1, uvs.xy);
#else
    half4 color0 = tex2D(Map2X1, uvs.xy);
#endif

#ifdef _POST_PARSE_	
    half4 color1 = h4tex2D(Map2X2, uvs.zw);
#else
    half4 color1 = tex2D(Map2X2, uvs.zw);
#endif

	Colour = lerp(color0, color1, 0.5f);

#if defined(POST_PROCESS_EFFECT_INDEX)
	_output.Colour.rgb = DoSimplePostProcessingEffect( Colour.rgb, _input.texCoords.xy );
	_output.Colour.a=Colour.a;
#else
	_output.Colour = Colour;
#endif

	return _output;
}

PSOUTPUT FragmentShader4XBlend( VSOUTPUT _input )
{
	PSOUTPUT _output;

	// Read textures
	float2	uv = _input.texCoord;
	half4	Colour;
#ifdef _POST_PARSE_
  	Colour = h4tex2D( Map4X, uv );
#else
  	Colour = tex2D( Map4X, uv );
#endif

#if defined(POST_PROCESS_EFFECT_INDEX)
	_output.Colour = DoSimplePostProcessingEffect( Colour.rgb, _input.texCoord );
#else
	_output.Colour = Colour;
#endif

	return _output;
}

PSOUTPUT FragmentShader2XBlend( VSOUTPUT2X _input )
{
	PSOUTPUT _output;
	half4	Colour;

	float4	uvs = _input.texCoords;

#ifdef _POST_PARSE_
    half4 color0 = h4tex2D(Map2X1, uvs.xy);
#else
    half4 color0 = tex2D(Map2X1, uvs.xy);
#endif

#ifdef _POST_PARSE_	
    half4 color1 = h4tex2D(Map2X2, uvs.zw);
#else
    half4 color1 = tex2D(Map2X2, uvs.zw);
#endif

	Colour = lerp(color0, color1, 0.5f);
	
#if defined(POST_PROCESS_EFFECT_INDEX)
	_output.Colour = DoSimplePostProcessingEffect( Colour.rgb, _input.texCoords.xy );
#else
	_output.Colour = Colour;
#endif

	return _output;
}

PSOUTPUT FragmentShaderClear( VSCLEAROUTPUT _input )
{
	PSOUTPUT _output;
	
	_output.Colour=ClearColour;
	
	return _output;
}

//-----------------------------------------------------------------------
//
// Technique(s)
//

technique MSAA4X
{
	pass Pass0
	{
#ifdef _PS3_
		ZWriteEnable=false;
		ZEnable=false;
		CullFaceEnable=false;
		AlphaBlendEnable=false;
		AlphaTestEnable = false;
		StencilTestEnable = false;
		VertexShader = compile sce_vp_rsx MSAAVertexShader();
		PixelShader = compile sce_fp_rsx FragmentShader4X();
#endif
	}
}

technique MSAA2X
<
	int regcount=11;
>
{
	pass Pass0
	{
#ifdef _PS3_
		ZWriteEnable=false;
		ZEnable=false;
		CullFaceEnable=false;
		AlphaBlendEnable=false;
		AlphaTestEnable = false;
		StencilTestEnable = false;
		VertexShader = compile sce_vp_rsx MSAAVertexShader2X();
		PixelShader = compile sce_fp_rsx FragmentShader2X();
#endif
	}
}

technique MSAA4XBlend
{
	pass Pass0
	{
#ifdef _PS3_
		ZWriteEnable=false;
		ZEnable=false;
		CullFaceEnable=false;
		AlphaBlendEnable=true;
		AlphaTestEnable = false;
		StencilTestEnable = false;
		BlendFunc=int2(SrcAlpha, OneMinusSrcAlpha);
		BlendEquation=int(FuncAdd);
		VertexShader = compile sce_vp_rsx MSAAVertexShader();
		PixelShader = compile sce_fp_rsx FragmentShader4XBlend();
#endif
	}
}

technique MSAA2XBlend
{
	pass Pass0
	{
#ifdef _PS3_
		ZWriteEnable=false;
		ZEnable=false;
		CullFaceEnable=false;
		AlphaBlendEnable=true;
		AlphaTestEnable = false;
		StencilTestEnable = false;
		BlendFunc=int2(SrcAlpha, OneMinusSrcAlpha);
		BlendEquation=int(FuncAdd);
		VertexShader = compile sce_vp_rsx MSAAVertexShader2X();
		PixelShader = compile sce_fp_rsx FragmentShader2XBlend();
#endif
	}
}

technique Clear
{
	pass Pass0
	{
#ifdef _PS3_
		ZWriteEnable=false;
		ZEnable=false;
		CullFaceEnable=false;
		AlphaBlendEnable=false;
		AlphaTestEnable = false;
		StencilTestEnable = false;
		VertexShader = compile sce_vp_rsx MSAAClearVertexShader();
		PixelShader = compile sce_fp_rsx FragmentShaderClear();
#endif
	}
}
